<?php
namespace App\Models;

use App\Models\Amenity;
use App\Models\Dashboard\Admin\Admin;
use App\Models\Feature;
use App\Models\FurnishingStatus;
use App\Models\Neighborhood;
use App\Models\PropertyStatus;
use App\Models\PropertyType;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

class Property extends Model
{
    use HasFactory;

    protected $guarded = [];

    protected static function booted()
    {
        static::creating(function ($model) {
            if (empty($model->uuid)) {
                $model->uuid = Str::uuid()->toString();
            }
        });
    }

    public function admin()
    {
        return $this->belongsTo(Admin::class)->select('id', 'full_name');
    }

    public function city()
    {
        return $this->belongsTo(City::class);
    }

    public function neighborhood()
    {
        return $this->belongsTo(Neighborhood::class);
    }

    public function type()
    {
        return $this->belongsTo(PropertyType::class, 'property_type_id');
    }

    public function status()
    {
        return $this->belongsTo(PropertyStatus::class, 'property_status_id');
    }

    public function furnishingStatus()
    {
        return $this->belongsTo(FurnishingStatus::class, 'furnishing_status_id');
    }

    public function features()
    {
        return $this->belongsToMany(Feature::class, 'property_feature');
    }

    public function amenities()
    {
        return $this->belongsToMany(Amenity::class, 'property_amenity');
    }

    public function attachments()
    {
        return $this->hasMany(PropertyAttachment::class);
    }

    public function units()
    {
        return $this->hasMany(PropertyUnits::class);
    }

    public function scopeFilter($query, $filters = [])
    {

        if (! empty($filters['title'])) {
            $normalized = normalizeArabic($filters['title']);

            // حول المسافات لكلمة + كلمة للبحث boolean mode
            $booleanSearch = str_replace(' ', ' ', $normalized);

            $query->whereRaw(
                "MATCH(title_normalized, description_normalized) AGAINST(? IN BOOLEAN MODE)",
                [$booleanSearch]
            );
        }

        if (! empty($filters['neighborhood_id'])) {
            $query->where('neighborhood_id', $filters['neighborhood_id']);
        }

        if (! empty($filters['property_type_id'])) {
            $query->where('property_type_id', $filters['property_type_id']);
        }

        if (! empty($filters['property_status_id'])) {
            $query->where('property_status_id', $filters['property_status_id']);
        }

        if (! empty($filters['furnishing_status_id'])) {
            $query->where('furnishing_status_id', $filters['furnishing_status_id']);
        }

        if (! empty($filters['price_min'])) {
            $query->where('price', '>=', $filters['price_min']);
        }

        if (! empty($filters['price_max'])) {
            $query->where('price', '<=', $filters['price_max']);
        }

        // if (! empty($filters['bathrooms'])) {
        //     $query->where('bathrooms', $filters['bathrooms']);
        // }

        // if (! empty($filters['bedrooms'])) {
        //     $query->where('bedrooms', $filters['bedrooms']);
        // }

        // ✅ هنا نعدل البحث في الغرف
        if (! empty($filters['bedrooms'])) {
            $query->where(function ($q) use ($filters) {
                $q->where('bedrooms', $filters['bedrooms'])
                    ->orWhereHas('units', function ($sub) use ($filters) {
                        $sub->where('bedrooms', $filters['bedrooms']);
                    });
            });
        }

        // ✅ وهنا للحمامات
        if (! empty($filters['bathrooms'])) {
            $query->where(function ($q) use ($filters) {
                $q->where('bathrooms', $filters['bathrooms'])
                    ->orWhereHas('units', function ($sub) use ($filters) {
                        $sub->where('bathrooms', $filters['bathrooms']);
                    });
            });
        }

        return $query;
    }

}
